﻿/*
	Name:			XenoN Core Application Component Header
	Version:		2.6
	Update:			2011-02-24
	Copyright:		Copyright © 2007-2011 XenoN Core by PsichiX. All rights reserved.
	Author:			PsichiX
	Website:		http://www.xenon.psichix.com
	Description:	XenoN Core Framework Application Component Header
*/

/*
	== EN ==
	This file is part of Xenon Core Framework.
	You may distribute it further, but you can not modify it.
	Please do not use in modified form.
	The principles of XenoN Core License available in the file LICENSE_CORE_EN.TXT or visit: http://www.xenon.psichix.com.

	== PL ==
	Ten plik jest czescia XenoN Core Framework.
	Mozesz go rozpowszechniac dalej, jednak nie mozesz go modyfikowac.
	Nalezy uzytkowac w nie zmodyfikowanej formie.
	Nalezy przestrzegac zasad Licencji XenoN Core dostepnej w pliku LICENSE_CORE_PL.TXT oraz na stronie: http://www.xenon.psichix.com.
*/

#ifndef XE_COMPONENT_APP_H
#define XE_COMPONENT_APP_H

#include "../System/XenonCoreFramework.h"
#ifdef XE_COMPILE_ECHO
#include "XeComponentAudio.h"
#endif /* XE_COMPILE_ECHO */
#include "XeComponent2D.h"
#include <Shlobj.h>

namespace XeCore
{
//! Przestrzen nazw komponentow
namespace Com
{

class C_ALARM;
class C_PARTICLES;
class C_APPLICATION;

typedef	XE_ELEMENT_POINTER<XE_WINDOW>		C_ELM_WINDOW;
#ifdef XE_COMPILE_PHOTON
typedef	XE_ELEMENT_POINTER<XE_SPRITE>		C_ELM_SPRITE;
#endif /* XE_COMPILE_PHOTON */
typedef	XE_ELEMENT_POINTER<XE_ACTOR*>		C_ELM_ACTOR;
typedef	XE_ELEMENT_POINTER<void*>			C_ELM_OBJECT;
typedef	XE_ELEMENT_POINTER<XE_RESOURCE>		C_ELM_RESOURCE;
#ifdef XE_COMPILE_PHOTON
typedef	XE_ELEMENT_POINTER<XE_FONT>			C_ELM_FONT;
#endif /* XE_COMPILE_PHOTON */
typedef	XE_ELEMENT_POINTER<C_ALARM>			C_ELM_ALARM;
typedef	XE_ELM_VOIDPROC						C_ELM_SCENE;

//! Klasa alarmu (timera).
class C_ALARM
{
public:
	//! Czas aktualny.
	float Time;
	//! Wskaźnik na funkcję która ma się wykonywać po odliczaniu czasu.
	void (*Action)();
	//! Konstruktor obiektu.
	C_ALARM();
	/*! Zdarzenie postępu.
	\param factor Wspolczynnik.
	*/
	void Progress(float factor);
};
//! Definicja elementu alarmu.
typedef XE_ELEMENT_POINTER<C_ALARM> C_ELM_ALARM;

#ifdef XE_CANIMP

C_ALARM::C_ALARM()
{
	Time = 0;
	Action = NULL;
}

void C_ALARM::Progress(float factor)
{
	if(Time>0)
	{
		Time-=factor;
		if(Time<=0)
		{
			Time=0;
			if(Action!=NULL)Action();
		}
	}
}

#endif /* XE_CANIMP */

#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH

//! Rozszerzona klasa czastek
class C_PARTICLES : public XE_PARTICLE_STREAM
{
public:
	//! Ustala czy czasteczki maja byc rysowane pod obiektami (domyslnie rysowane sa nad obiektami).
	bool Below;
	//! Wskaznik na aktora z ktorym sa polaczone czasteki (sa rysowane nad lub pod danym aktorem).
	XE_ACTOR* Actor;
	//! Wskaznik na funkcje bedaca programem zastepczym aktualizacji czastek.
	void (*Program)(XE_PARTICLE_UNIT*);
	//! Konstruktor.
	C_PARTICLES();
};

#ifdef XE_CANIMP

C_PARTICLES::C_PARTICLES()
{
	Below=false;
	Actor=NULL;
	Program=NULL;
}

#endif /* XE_CANIMP */

#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */

//! Klasa managera aplikacji
class C_APPLICATION
{
public:
	typedef XE_ELEMENTS<XE_PAIR<XE_STRING,int> > FILELIST;
private:
	unsigned int LastUnique;
	int DisplayWidth;
	int DisplayHeight;
	float DisplayAspect;
	XE_CONTAINER<XE_ACTOR*> Unactive;
	XE_CONTAINER<XE_ACTOR*> ToRegister;
	XE_ELEMENTS<XE_ACTOR*> ToUnregister;
	static int SortByOrder(XE_ACTOR** a,XE_ACTOR** b);
public:
	bool AutoSessionStage;
	bool AutoActorStage;
	bool AutoParticleStage;
	bool AutoDnaStage;
	bool AutoGuiStage;
	bool AutoAlarmStage;
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
										//! Kamera.
	XE_CAMERA							Camera;
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
#ifdef XE_COMPILE_ECHO
										//! Audio.
	C_AUDIO								Audio;
#endif /* XE_COMPILE_ECHO */
										//! Sceny.
	XE_SCENES							Scenes;
										//! Okna.
	XE_CONTAINER<XE_WINDOW>				Windows;
#ifdef XE_COMPILE_PHOTON
										//! Sprajty.
	XE_CONTAINER<XE_SPRITE>				Sprites;
										//! Modele.
	XE_CONTAINER<XE_MODEL>				Models;
#endif /* XE_COMPILE_PHOTON */
										//! Aktorzy.
	XE_CONTAINER<XE_ACTOR*>				Actors;
										//! Obiekty.
	XE_CONTAINER<void*>					Objects;
										//! Zasoby.
	XE_CONTAINER<XE_RESOURCE>			Resources;
#ifdef XE_COMPILE_PHOTON
										//! Czcionki.
	XE_CONTAINER<XE_FONT>				Fonts;
#endif /* XE_COMPILE_PHOTON */
										//! Alarmy.
	XE_CONTAINER<C_ALARM>				Alarms;
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
										//! Viewy (obszary widoku).
	XE_CONTAINER<C_VIEW>				Views;
										//! Aktywny view.
	C_ELM_VIEW							ViewActive;
										//! Czasteczki.
	XE_CONTAINER<C_PARTICLES>			Particles;
#endif /* XE_COMPILE_CORE_MATH */
										//! Graficzny interfejs uzytkownika.
	XE_GUI								GUI;
										//! Efekty (shadery).
	XE_CONTAINER<XE_EFFECT>				Effects;
#ifdef XE_COMPILE_CORE_MATH
										//! Powierzchnie rysowania (bufory ramki).
	XE_CONTAINER<XE_CANVAS>				Canvases;
#endif /* XE_COMPILE_CORE_MATH */
#ifdef XE_COMPILE_CORE_INTUICIO
#ifdef XE_COMPILE_CORE_MATH
										//! Techniki efektow.
	XE_CONTAINER<XE_TECHNIQUE>			Techniques;
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_CORE_INTUICIO */
#ifdef XE_COMPILE_CORE_DNA
#ifdef XE_COMPILE_CHAOS
#ifdef XE_COMPILE_CORE_MATH
										//! Modele DNA.
	XE_CONTAINER<XE_DNA_MODEL>			DNA;
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_CHAOS */
#endif /* XE_COMPILE_CORE_DNA */
#endif /* XE_COMPILE_PHOTON */
#ifdef XE_COMPILE_ETHER
										//! Sesje sieciowe.
	XE_CONTAINER<XE_SESSION>			Sessions;
#endif /* XE_COMPILE_PHOTON */
										//! Globalna procedura aktualizacji.
	XE_VOIDPROC							GlobalUpdateProc;
										//! Globalna procedura renderowania (przed renderingiem aktorow).
	XE_VOIDPROC							GlobalPreRenderProc;
										//! Globalna procedura renderowania (po renderingu aktorow).
	XE_VOIDPROC							GlobalPostRenderProc;
										//! Konstruktor domyslny.
										C_APPLICATION();
										//! Destruktor.
										~C_APPLICATION();
										/*! Tworzy obiekt aplikacji.
										\return Wskaznik na obiekt aplikacji.
										*/
	static C_APPLICATION*				MakeInstance();
										/*! Czysci kontenery. */
	void								Cleanup();
										/*! Otwiera lub tworzy klucz rejestru.
										\param root Uchwyt klucza bazowego.
										\param name Nazwa klucza.
										\param mode Tryb otwarcia. Dostepne wartosci:<br>
										XEF_OPEN: Otwarcie.<br>
										XEF_CREATE: Utworzenie i otwarcie.
										\return Uchwyt otwartego klucza jesli operacja powiodla sie, lub wartosc NULL jesli nie.
										*/
	HKEY								RegistryKeyOpen(HKEY root,char* name,XEF_ESTATE mode=XEF_OPEN);
										/*! Usuniecie klucza.
										\param key Uchwyt klucza.
										\param name Nazwa klucza.
										\return true jesli operacja powiodla sie.
										*/
	bool								RegistryKeyDelete(HKEY key,char* name);
										/*! Ustalenie wartosci klucza.
										\param key Uchwyt klucza.
										\param name Nazwa wartosci.
										\param type Typ wartosci. Dostepne wartosci:<br>
										REG_BINARY: Ciag binarny.<br>
										REG_DWORD: Liczba 32-bit bez znaku (unsigned int).<br>
										REG_SZ: String.<br>
										REG_MULTI_SZ: Tablica stringow.
										\param data Wskaznik na dane.
										\param size Rozmiar bufora danych.
										\return true jesli operacja powiodla sie.
										*/
	bool								RegistryValSet(HKEY key,char* name,XE_DWORD type,void* data,unsigned int size);
										/*! Usuniecie wartosci.
										\param key Uchwyt klucza.
										\param name Nazwa wartosci.
										\return true jesli operacja powiodla sie.
										*/
	bool								RegistryValDelete(HKEY key,char* name);
										/*! Pobranie wartosci klucza.
										\param key Uchwyt klucza.
										\param name Nazwa wartosci.
										\param type Wskaznik na typ wartosci.
										\param data Wskaznik na dane.
										\param size Wskaznik na rozmiar bufora danych.
										\return true jesli operacja powiodla sie.
										*/
	bool								RegistryValGet(HKEY key,char* name,XE_DWORD* type,void* data,unsigned int* size);
										/*! Sprawdza czy wartosc istnieje.
										\param key Uchwyt klucza.
										\param name Nazwa wartosci.
										\return true jesli operacja powiodla sie.
										*/
	bool								RegistryValExists(HKEY key,char* name);
										/*! Tworzy w rejestrze skojarzenie plikow z danym programem.
										\param name Nazwa skojarzenia.
										\param ext Rozszerzenie plikow.
										\param exe Sciezka do pliku EXE programu.
										\param icon Sciezka do pliku ikony.
										\return true jesli operacja powiodla sie.
										*/
	bool								AssociateFile(char* name,char* ext,char* exe,char* icon);
										/*! Zwraca szerokosc wyswietlacza w pikselach.
										\return Szerokosc wyswietlacza.
										*/
	int									GetDisplayWidth();
										/*! Zwraca wysokosc wyswietlacza w pikselach.
										\return Wysokosc wyswietlacza.
										*/
	int									GetDisplayHeight();
										/*! Zwraca aspekt wyswietlacza.
										\return Aspekt wyswietlacza.
										*/
	float								GetDisplayAspect();
#ifdef XE_COMPILE_PHOTON
										/*! Ustawia/odswierza wlasciwosci render targetu okna.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										*/
	void								OnWindowSetup(char* name,unsigned int id=0);
#endif /* XE_COMPILE_PHOTON */
										/*! Centrowanie okna.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										*/
	void								WindowCentralize(char* name,unsigned int id=0);
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
										/*! Aktywuje view pobierajac wlasciwosci danego okna.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										*/
	void								WindowViewActivate(char* name,unsigned int id=0);
#endif /* XE_COMPILE_CORE_MATH */
										/*! Pionowa synchronizacja obrazu.
										\param mode Tryb wlaczony (true) lub wylaczony (false).
										*/
	void								Vsync(bool mode);
#endif /* XE_COMPILE_PHOTON */
										/*! Zwraca pozycje X myszki.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\return Pozycja X.
										*/
	int									MouseGetX(char* name=0,unsigned int id=0);
										/*! Zwraca pozycje X myszki w viewie.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\param view Nazwa viewa, lub NULL dla bierzacego viewa.
										\param viewid Identyfikator viewa.
										\return Pozycja X.
										*/
	int									MouseGetViewX(char* name=0,unsigned int id=0,char* view=0,unsigned int viewid=0);
										/*! Zwraca pozycje Y myszki.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\return Pozycja Y.
										*/
	int									MouseGetY(char* name=0,unsigned int id=0);
										/*! Zwraca pozycje Y myszki w viewie.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\param view Nazwa viewa, lub NULL dla bierzacego viewa.
										\param viewid Identyfikator viewa.
										\return Pozycja Y.
										*/
	int									MouseGetViewY(char* name=0,unsigned int id=0,char* view=0,unsigned int viewid=0);
										/*! Zwraca wektor pozycji myszki.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\return Wektor pozycji.
										*/
	XE_VECTOR							MouseGetPos(char* name=0,unsigned int id=0);
										/*! Zwraca wektor pozycji myszki w viewie.
										\param name Nazwa okna wzgledem ktorego zwraca pozycje. Jesli nie podamy nazwy, zwraca pozycje bezwzgledna.
										\param id Identyfikator okna.
										\param view Nazwa viewa, lub NULL dla bierzacego viewa.
										\param viewid Identyfikator viewa.
										\return Wektor pozycji.
										*/
	XE_VECTOR							MouseGetViewPos(char* name=0,unsigned int id=0,char* view=0,unsigned int viewid=0);
										/*! Sprawdza czy pozycja myszy znajduje sie w danym obszarze.
										\param x1 Poczatkowa pozycja X.
										\param y1 Poczatkowa pozycja Y.
										\param x2 Koncowa pozycja X.
										\param y2 Koncowa pozycja Y.
										\param name Nazwa okna wzgledem ktorego sprawdza. Jesli nie podamy nazwy, sprawdza wzgledem wyswietlacza.
										\param id Identyfikator okna.
										\return true jesli operacja powiodla sie.
										*/
	bool								MouseInside(int x1,int y1,int x2,int y2,char* name=0,unsigned int id=0);
										/*! Sprawdza czy przycisk myszy nie jest trzymany nad sprajtem na danej pozycji.
										\param winname Nazwa okna.
										\param sprname Nazwa sprajta.
										\param sprpos Pozycja na ktorej znajduje sie sprite.
										\param button Identyfikator przycisku myszy.
										\return true jesli przycisk myszy nie jest trzymany nad sprajtem, lub false jesli nie.
										*/
	bool								MouseHoldSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button=XE_MOUSE_LEFT);
										/*! Sprawdza czy przycisk myszy nie jest wcisniety nad sprajtem na danej pozycji.
										\param winname Nazwa okna.
										\param sprname Nazwa sprajta.
										\param sprpos Pozycja na ktorej znajduje sie sprite.
										\param button Identyfikator przycisku myszy.
										\return true jesli przycisk myszy nie jest wcisniety nad sprajtem, lub false jesli nie.
										*/
	bool								MousePressSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button=XE_MOUSE_LEFT);
										/*! Sprawdza czy przycisk myszy nie jest puszczony nad sprajtem na danej pozycji.
										\param winname Nazwa okna.
										\param sprname Nazwa sprajta.
										\param sprpos Pozycja na ktorej znajduje sie sprite.
										\param button Identyfikator przycisku myszy.
										\return true jesli przycisk myszy nie jest puszczony nad sprajtem, lub false jesli nie.
										*/
	bool								MouseReleaseSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button=XE_MOUSE_LEFT);
										/*! Wyswietla okno wiadomosci.
										\param text Tresc wiadomosci.
										\param title Tytul.
										\param flags Flagi opcji wiadomosci.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										\return Kod rezultatu.
										*/
	int									Message(char* text,char* title=0,unsigned int flags=0,char* name=0,unsigned int id=0);
										/*! Otwiera plik lub strone www.
										\param file Adres pliku badz strony www.
										\param parameters Parametry.
										\param initdir Folder domyslny z ktorego zostanie otwarty plik.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										*/
	void								Open(char* file,char* parameters=0,char* initdir=0,char* name=0,unsigned int id=0);
										/*! Pokazuje okno dialogowe wyboru nazwy pliku do otwarcia badz zapisu.
										\param type Typ okna. Dostepne wartosci:<br>
										XEF_OPEN: Otwarcie pliku.<br>
										XEF_SAVE: Zapis pliku.
										\param extensions Akceptowane rozszerzenia plikow. Przyklad: "XET Texture|*.xet|Intuicio Program|*.itc".
										\param title Tytul okna.
										\param initdir Wskaznik na sciezke katalogu domyslnego.
										\param flags Flagi.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										\return String z wynikowa sciezka dostepu.
										*/
	XE_STRING							GetFilename(XEF_ESTATE type=XEF_OPEN,char* extensions=0,char* title=0,char* initdir=0,unsigned int flags=0,char* name=0,unsigned int id=0);
										/*! Listuje wszystkie pliki (i foldery) w danym folderze, według filtru nazwy.
										\param list Referencja na kontener listy plikow.
										\param path Sciezka do folderu wraz z filtrem nazw, np: "\*.xet".
										*/
	void								ListFiles(FILELIST& list,char* path);
										/*! Pokazuje okno dialogowe wyboru folderu.
										\param title Tytul okna.
										\param initdir Wskaznik na sciezke katalogu domyslnego.
										\param flags Flagi.
										\param name Nazwa okna.
										\param id Identyfikator okna.
										*/
	XE_STRING							GetDirname(char* title=0,char* initdir=0,unsigned int flags=0,char* name=0,unsigned int id=0);
										/*! Prowadzi aktora do danego punktu.
										\param actor Wskaznik na aktora.
										\param point Wektor punktu docelowego.
										\param speed Dlugosc kroku.
										*/
	void								MoveTo(XE_ACTOR* actor,XE_VECTOR point,double speed);
										/*! Przesuwa dana pozycje w kierunku danego punktu.
										\param pos Wektor pozycji poczatkowej.
										\param point Wektor punktu docelowego.
										\param speed Dlugosc kroku.
										\return Wektor pozycji po przesunieciu.
										*/
	XE_VECTOR							MoveTo(XE_VECTOR pos,XE_VECTOR point,double speed);
										/*! Prowadzi aktora do danego punktu, hamujac w danym zasiegu.
										\param actor Wskaznik na aktora.
										\param point Wektor punktu docelowego.
										\param speed Dlugosc kroku.
										\param distance Dlugosc zasiegu hamowania.
										*/
	void								TorqueMoveTo(XE_ACTOR* actor,XE_VECTOR point,double speed,double distance);
										/*! Przesuwa dana pozycje w kierunku danego punktu, hamujac w danym zasiegu.
										\param pos Wektor pozycji poczatkowej.
										\param point Wektor punktu docelowego.
										\param speed Dlugosc kroku.
										\param distance Dlugosc zasiegu hamowania.
										\return Wektor pozycji po przesunieciu.
										*/
	XE_VECTOR							TorqueMoveTo(XE_VECTOR pos,XE_VECTOR point,double speed,double distance);
										/*! Losuje i zwraca jedna z podanych wartosci danego typu.
										\param count Ilosc wartosci.
										\param ... Kolejne wartosci danego typu.
										\return Wylosowana wartosc.
										*/
										template<typename T>
	T									Choose(int count,...);
										/*! Konczy odtwarzanie bierzacej i przechodzi do wykonywania danej sceny.
										\param name Nazwa sceny.
										\param id Identyfikator.
										\return true jesli operacja powiodla sie.
										*/
	bool								GoToScene(char* name,unsigned int id=0);
										/*! Konczy odtwarzanie bierzacej i przechodzi do wykonywania danej sceny.
										\param scene Element sceny.
										\return true jesli operacja powiodla sie.
										*/
	bool								GoToScene(C_ELM_SCENE scene);
										/*! Przechodzi do nastepnej sceny. */
	bool								NextScene();
										/*! Przechodzi do poprzedniej sceny. */
	bool								PrevScene();
										/*! Konczy gre. */
	void								Exit();
#ifdef XE_COMPILE_PHOTON
										/*! Ustala kolor rysowania.
										\param color Kolor hexadecymalny.
										*/
	void								SetColor(int color);
	void								SetColor(XE_VECTOR color);
										/*! Czysci okno na dany kolor.
										\param color Kolor hexadecymalny.
										*/
	void								ClearSceneColor(int color);
	void								ClearSceneColor(XE_VECTOR color);
										/*! Ustawia tryb mieszania kolorow.
										\param src Zrodlowy.
										\param dst Docelowy.
										\param equation Rownanie mieszania kolorow.
										*/
	void								Blend(XE_ESTATE src,XE_ESTATE dst,XE_ESTATE equation=XE_RENDER_BLEND_ADD);
										/*! Ustawia domyslny tryb mieszania kolorow. */
	void								Unblend();
										/*! Sprawdza wystapienie kolizji z innymi aktorami.
										\param type Typ kolizji. Dostepne wartosci:<br>
										XEF_BOX: Kolizje bounding box.<br>
										XEF_CIRCLE: Kolizje okregow.<br>
										XEF_SHAPE: Kolizje ksztaltow.
										\param act Wskaznik na aktora 2D.
										\param desc Opis aktora.
										\param relative Wektor przesuniecia.
										\return Wskaznik na aktora 2D z ktorym nastapila kolizja lub wartosc NULL jesli nie wykryto zadnej kolizji.
										*/
	XE_ACTOR2D*							Collision2D(XEF_ESTATE type,XE_ACTOR2D* act,char* desc=0,XE_VECTOR relative=XE_VECTOR());
										/*! Sprawdza wystapienie kolizji z innymi aktorami i dodaje ich do listy.
										\param list Referencja na kontener elementow do ktorego dodamy aktorow.
										\param type Typ kolizji. Dostepne wartosci:<br>
										XEF_BOX: Kolizje bounding box.<br>
										XEF_CIRCLE: Kolizje okregow.<br>
										XEF_SHAPE: Kolizje ksztaltow.
										\param act Wskaznik na aktora 2D.
										\param desc Opis aktora.
										\param relative Wektor przesuniecia.
										*/
	void								Collision2Dlist(XE_ELEMENTS<XE_ACTOR2D*>& list,XEF_ESTATE type,XE_ACTOR2D* act,char* desc=0,XE_VECTOR relative=XE_VECTOR());
#endif /* XE_COMPILE_PHOTON */
										/*! Tworzy i zwraca unikatowy identyfikator.
										\return Unikatowy identyfikator.
										*/
	unsigned int						MakeUnique();
										/*! Sortuje aktorów. */
	void								SortActors();
										/*! Rejestruje nowe obiekty z kolejki do dodania. */
	void								Register();
										/*! Odrejestrowuje obiekty z kolejki do usuniecia. */
	void								Unregister();
										/*! Rejestruje unikalnego aktora.
										\param name Nazwa aktora.
										\param newactor Wskaznik na nowego aktora.
										\return Wskaznik na utworzonego aktora, badz wartosc NULL jesli operacja nie powiodla sie.
										*/
	XE_ACTOR*							RegisterActor(char* name,XE_ACTOR* newactor);
										/*! Wyrejestrowuje aktora.
										\param actor Wskaznik na aktora do wyrejestrowania.
										*/
	void								UnregisterActor(XE_ACTOR* actor);
										/*! Nadajemy 123-znakowy opis obiektu.
										\param actor Wskaznik na obiekt aktora.
										\param text Opis tekstowy majacy nie wiecj niz 123 znaki.
										*/
	void								Describe(XE_ACTOR* actor,char* text);
										/*! Sprawdzamy czy opis obiektu jest identyczny z podanym w argumencie.
										\param actor Wskaznik na obiekt aktora.
										\param text Opis tekstowy majacy nie wiecj niz 123 znaki.
										\return true jesli operacja powiodla sie.
										*/
	bool								IsDescribedAs(XE_ACTOR* actor,char* text);
										/*! Aktualizuje alarmy. */
	void								AlarmsProgress();
										/*! Aktualizuje aktorow. */
	void								Update();
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
										/*! Renderuje aktorow. */
	void								Render();
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
										/*! Aktualizacja postepu sceny. */
	void								Progress();
};

#ifdef XE_CANIMP

int C_APPLICATION::SortByOrder(XE_ACTOR** a,XE_ACTOR** b)
{
	if((*a)->Order==(*b)->Order)return(0);
	if((*a)->Order>(*b)->Order)return(1);
	return(-1);
}

C_APPLICATION::C_APPLICATION()
{
	LastUnique=0;
	XE_WINDOW temp;
	temp.FromDisplay();
	DisplayWidth=temp.Width;
	DisplayHeight=temp.Height;
	DisplayAspect=(float)DisplayWidth/(float)DisplayHeight;
	AutoSessionStage = true;
	AutoActorStage = true;
	AutoParticleStage = true;
	AutoDnaStage = true;
	AutoGuiStage = true;
	AutoAlarmStage = true;
}

C_APPLICATION::~C_APPLICATION()
{
	Cleanup();
	Scenes.Clear();
	Windows.Clear();
}

C_APPLICATION* C_APPLICATION::MakeInstance()
{
	return(new C_APPLICATION());
}

void C_APPLICATION::Cleanup()
{
#ifdef XE_COMPILE_ECHO
	Audio.MusicClean();
	Audio.SoundClean();
#endif /* XE_COMPILE_ECHO */
#ifdef XE_COMPILE_PHOTON
	XE_FOREACH(XE_MODEL,Models,elm)elm.iPointer()->Cleanup();
	Models.Clear();
	XE_FOREACH(XE_SPRITE,Sprites,elm)elm.iPointer()->Cleanup();
	Sprites.Clear();
#endif /* XE_COMPILE_PHOTON */
	XE_FOREACH(XE_ACTOR*,Actors,elm)if(elm.iClone()){delete elm.iClone();}
	Actors.Clear();
	XE_FOREACH(XE_ACTOR*,ToRegister,elm)if(elm.iClone())delete elm.iClone();
	ToRegister.Clear();
	XE_FOREACH(XE_ACTOR*,ToUnregister,elm)if(elm.iClone())delete elm.iClone();
	ToUnregister.Clear();
	XE_FOREACH(void*,Objects,elm)if(elm.iClone())delete elm.iClone();
	Objects.Clear();
	Resources.Clear();
#ifdef XE_COMPILE_PHOTON
	XE_FOREACH(XE_FONT,Fonts,elm)elm.iPointer()->Cleanup();
	Fonts.Clear();
	GUI.Cleanup();
#endif /* XE_COMPILE_PHOTON */
	Alarms.Clear();
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
	Views.Clear();
	XE_FOREACH(C_PARTICLES,Particles,elm)elm.iPointer()->Free();
	Particles.Clear();
#endif /* XE_COMPILE_CORE_MATH */
	XE_FOREACH(XE_EFFECT,Effects,elm)elm.iPointer()->Cleanup();
	Effects.Clear();
#ifdef XE_COMPILE_CORE_MATH
	XE_FOREACH(XE_CANVAS,Canvases,elm)elm.iPointer()->Destroy();
	Canvases.Clear();
#ifdef XE_COMPILE_CORE_INTUICIO
	XE_FOREACH(XE_TECHNIQUE,Techniques,elm)elm.iPointer()->FreeProgram();
	Techniques.Clear();
#endif /* XE_COMPILE_CORE_INTUICIO */
#endif /* XE_COMPILE_CORE_MATH */
#ifdef XE_COMPILE_CORE_DNA
#ifdef XE_COMPILE_CHAOS
#ifdef XE_COMPILE_CORE_MATH
	XE_FOREACH(XE_DNA_MODEL,DNA,elm)elm.iPointer()->Free();
	DNA.Clear();
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_CHAOS */
#endif /* XE_COMPILE_CORE_DNA */
#endif /* XE_COMPILE_PHOTON */
#ifdef XE_COMPILE_ETHER
	XE_FOREACH(XE_SESSION,Sessions,elm)elm.iPointer()->Exit();
	Sessions.Clear();
#endif /* XE_COMPILE_ETHER */
}

HKEY C_APPLICATION::RegistryKeyOpen(HKEY root,char* name,XEF_ESTATE mode)
{
	HKEY key;
	if(mode==XEF_OPEN)
		if(RegOpenKeyEx(root,name,0,KEY_ALL_ACCESS,&key)==ERROR_SUCCESS)return(key);
	if(mode==XEF_CREATE)
		if(RegCreateKeyEx(root,name,0,NULL,REG_OPTION_NON_VOLATILE,KEY_ALL_ACCESS,NULL,&key,NULL)==ERROR_SUCCESS)return(key);
	return(0);
}

bool C_APPLICATION::RegistryKeyDelete(HKEY key,char* name)
{
	if(RegDeleteKey(key,name)==ERROR_SUCCESS)
	return(true);
	return(false);
}

bool C_APPLICATION::RegistryValSet(HKEY key,char* name,XE_DWORD type,void* data,unsigned int size)
{
	if(RegSetValueEx(key,name,0,type,(BYTE*)data,size)==ERROR_SUCCESS)
	return(true);
	return(false);
}

bool C_APPLICATION::RegistryValDelete(HKEY key,char* name)
{
	if(RegDeleteValue(key,name)==ERROR_SUCCESS)
	return(true);
	return(false);
}

bool C_APPLICATION::RegistryValGet(HKEY key,char* name,XE_DWORD* type,void* data,unsigned int* size)
{
	if(RegQueryValueEx(key,name,0,(DWORD*)type,(BYTE*)data,(DWORD*)size)==ERROR_SUCCESS)
	return(true);
	return(false);
}

bool C_APPLICATION::RegistryValExists(HKEY key,char* name)
{
	if(RegQueryValueEx(key,name,0,0,0,0)==ERROR_SUCCESS)
	return(true);
	return(false);
}

bool C_APPLICATION::AssociateFile(char* name,char* ext,char* exe,char* icon)
{
	XE_STRING str;
	HKEY key=RegistryKeyOpen(HKEY_CLASSES_ROOT,ext,XEF_CREATE);
	if(key)
	if(RegistryValSet(key,"",REG_SZ,name,strlen(name)+1))
	{
		str.Format("%s\\DefaultIcon",name);
		key=RegistryKeyOpen(HKEY_CLASSES_ROOT,str.Get(),XEF_CREATE);
		if(key)
		if(RegistryValSet(key,"",REG_SZ,icon,strlen(icon)+1))
		{
			str.Format("%s\\shell\\open\\command",name);
			key=RegistryKeyOpen(HKEY_CLASSES_ROOT,str.Get(),XEF_CREATE);
			if(key)
			if(RegistryValSet(key,"",REG_SZ,exe,strlen(exe)+1))
			return(true);
		}
	}
	return(false);
}

int C_APPLICATION::GetDisplayWidth()
{
	return(DisplayWidth);
}

int C_APPLICATION::GetDisplayHeight()
{
	return(DisplayHeight);
}

float C_APPLICATION::GetDisplayAspect()
{
	return(DisplayAspect);
}

#ifdef XE_COMPILE_PHOTON

void C_APPLICATION::OnWindowSetup(char* name,unsigned int id)
{
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(elm.IsEmpty())return;

	Photon::XeRenderTargetActivate(elm.iPointer()->GetTarget());
	XeExtensions();
	Photon::XeRenderVsync(XE_FALSE);

	XeSetState(XE_RENDER_TEXTURING_2D,XE_TRUE);
	XeSetState(XE_SCENE_COLOR,0.0,0.0,0.0,0.0);
	XeSetState(XE_COLOR,1.0,1.0,1.0,1.0);
	XeSetState(XE_RENDER_BLEND,XE_TRUE);
	XeSetState(XE_RENDER_BLEND_TYPE,XE_RENDER_BLEND_SRC_ALPHA,XE_RENDER_BLEND_ONE_MINUS_SRC_ALPHA);
}

#endif /* XE_COMPILE_PHOTON */

void C_APPLICATION::WindowCentralize(char* name,unsigned int id)
{
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(elm.IsEmpty())return;
	elm.iPointer()->X=DisplayWidth/2-elm.iPointer()->Width/2;
	elm.iPointer()->Y=DisplayHeight/2-elm.iPointer()->Height/2;
}

#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH

void C_APPLICATION::WindowViewActivate(char* name,unsigned int id)
{
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(elm.IsEmpty())return;
	Photon::XeRenderTargetActivate(*(XE_ELM_RENDERTARGET*)XeGetState(XE_RENDERTARGET));
	XeCameraOrtho(0,0,(int)elm.iPointer()->Width,(int)elm.iPointer()->Height,0);
}

#endif /* XE_COMPILE_CORE_MATH */

void C_APPLICATION::Vsync(bool mode)
{
	if(!Photon::XeRenderVsync(mode?XE_TRUE:XE_FALSE))
	{
		XE_WINDOW temp;
		if(mode&&temp.GetRefresh())
		XE_EVENTS::Use().FrameTimeLimit=1000.0f/(float)temp.GetRefresh();
		else
		XE_EVENTS::Use().FrameTimeLimit=0;
		XE_EVENTS::FrameTimeUpdate();
	}
}

#endif /* XE_COMPILE_PHOTON */

int C_APPLICATION::MouseGetX(char* name,unsigned int id)
{
	if(!name)return(XE_IO::MouseGetX());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())return(XE_IO::MouseGetWinX(elm.iPointer()));
	return(0);
}

int C_APPLICATION::MouseGetViewX(char* name,unsigned int id,char* view,unsigned int viewid)
{
	if(!name)return(XE_IO::MouseGetX());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())
	{
		int val=XE_IO::MouseGetWinX(elm.iPointer());
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
		C_ELM_VIEW v=(!view)?(ViewActive):(Views(XEF_FIND,view,viewid));
		if(!v.IsEmpty())val+=(int)v->ScenePortPos.X;
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
		return(val);
	}
	return(0);
}

int C_APPLICATION::MouseGetY(char* name,unsigned int id)
{
	if(!name)return(XE_IO::MouseGetY());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())return(XE_IO::MouseGetWinY(elm.iPointer()));
	return(0);
}

int C_APPLICATION::MouseGetViewY(char* name,unsigned int id,char* view,unsigned int viewid)
{
	if(!name)return(XE_IO::MouseGetY());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())
	{
		int val=XE_IO::MouseGetWinY(elm.iPointer());
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
		C_ELM_VIEW v=(!view)?(ViewActive):(Views(XEF_FIND,view,viewid));
		if(!v.IsEmpty())val+=(int)v->ScenePortPos.Y;
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
		return(val);
	}
	return(0);
}

XE_VECTOR C_APPLICATION::MouseGetPos(char* name,unsigned int id)
{
	if(!name)return(XE_IO::MouseGetPos());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())return(XE_VECTOR(XE_IO::MouseGetWinX(elm.iPointer()),XE_IO::MouseGetWinY(elm.iPointer())));
	return(0);
}

XE_VECTOR C_APPLICATION::MouseGetViewPos(char* name,unsigned int id,char* view,unsigned int viewid)
{
	if(!name)return(XE_IO::MouseGetPos());
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())
	{
		XE_VECTOR val=XE_VECTOR(XE_IO::MouseGetWinX(elm.iPointer()),XE_IO::MouseGetWinY(elm.iPointer()));
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
		C_ELM_VIEW v=(!view)?(ViewActive):(Views(XEF_FIND,view,viewid));
		if(!v.IsEmpty())
		{
			val.X+=v->ScenePortPos.X;
			val.Y+=v->ScenePortPos.Y;
		}
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
		return(val);
	}
	return(0);
}

bool C_APPLICATION::MouseInside(int x1,int y1,int x2,int y2,char* name,unsigned int id)
{
	if(!name)return(XeMouseCheckRectPos(x1,y1,x2,y2));
	C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
	if(!elm.IsEmpty())return(XeMouseCheckRectWinPos(elm.iPointer(),x1,y1,x2,y2));
	return(0);
}

bool C_APPLICATION::MouseHoldSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button)
{
	if(XE_IO::MouseGet(button))
		return(RectCollision(MouseGetPos(winname),XE_VECTOR(-Sprites(sprname)->Xoffset,-Sprites(sprname)->Yoffset),XE_VECTOR(-Sprites(sprname)->Xoffset+Sprites(sprname)->Width,-Sprites(sprname)->Yoffset+Sprites(sprname)->Height)));
	return(false);
}

bool C_APPLICATION::MousePressSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button)
{
	if(XE_IO::MouseGetPressed(button))
		return(RectCollision(MouseGetPos(winname),XE_VECTOR(-Sprites(sprname)->Xoffset,-Sprites(sprname)->Yoffset),XE_VECTOR(-Sprites(sprname)->Xoffset+Sprites(sprname)->Width,-Sprites(sprname)->Yoffset+Sprites(sprname)->Height)));
	return(false);
}

bool C_APPLICATION::MouseReleaseSprite(char* winname,char* sprname,XE_VECTOR sprpos,int button)
{
	if(XE_IO::MouseGetReleased(button))
		return(RectCollision(MouseGetPos(winname),XE_VECTOR(-Sprites(sprname)->Xoffset,-Sprites(sprname)->Yoffset),XE_VECTOR(-Sprites(sprname)->Xoffset+Sprites(sprname)->Width,-Sprites(sprname)->Yoffset+Sprites(sprname)->Height)));
	return(false);
}

int C_APPLICATION::Message(char* text,char* title,unsigned int flags,char* name,unsigned int id)
{
	HWND temp=NULL;
	char* _temp="XenoN Core Application";
	if(title)_temp=title;
	if(name)
	{
		C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
		if(!elm.IsEmpty())
		{
			temp=elm.iPointer()->GetWindow();
			if(!title)_temp=elm.iPointer()->Name.Get();
		}
	}
	return(MessageBox(temp,text,_temp,flags));
}

void C_APPLICATION::Open(char* file,char* parameters,char* initdir,char* name,unsigned int id)
{
	if(!name)
		ShellExecute(NULL,"OPEN",file,parameters,initdir,SW_SHOWDEFAULT);
	else
	{
		C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
		if(!elm.IsEmpty())ShellExecute(elm.iPointer()->GetWindow(),"OPEN",file,parameters,initdir,SW_SHOWDEFAULT);
	}
}

XE_STRING C_APPLICATION::GetFilename(XEF_ESTATE type,char* extensions,char* title,char* initdir,unsigned int flags,char* name,unsigned int id)
{
	HWND window;
	if(!name)
		window=NULL;
	else
	{
		C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
		if(!elm.IsEmpty())
			window=elm.iPointer()->GetWindow();
		else
			window=NULL;
	}
	OPENFILENAME file={0};
	char dir[MAX_PATH]={0};
	char path[MAX_PATH]={0};
	if(!initdir)GetCurrentDirectory(MAX_PATH,dir);
	file.lStructSize=sizeof(OPENFILENAME);
	file.hwndOwner=window;
	file.lpstrFile=path;
	file.nMaxFile=MAX_PATH;
	if(!initdir)
		file.lpstrInitialDir=dir;
	else
		file.lpstrInitialDir=initdir;
	if(title)file.lpstrTitle=title;
	file.Flags=OFN_EXPLORER|OFN_NOCHANGEDIR;
	file.FlagsEx=0;
	if(flags&XEF_FLAG_OVERVRITE_PROMPT)file.Flags|=OFN_OVERWRITEPROMPT;
	if(flags&XEF_FLAG_MULTI_SELECT)file.Flags|=OFN_ALLOWMULTISELECT;
	if(flags&XEF_FLAG_NO_READONLY_SELECT)file.Flags|=OFN_HIDEREADONLY;
	if(flags&XEF_FLAG_NO_PLACES_BAR)file.FlagsEx|=OFN_EX_NOPLACESBAR;
	if(type==XEF_OPEN)file.Flags|=OFN_PATHMUSTEXIST|OFN_FILEMUSTEXIST;
	XE_STRING ext;
	if(extensions)
	{
		ext.Format("%s|",extensions);
		unsigned int len=ext.Length();
		for(unsigned int i=0;i<len;i++)
		if(ext.Get()[i]=='|')
		ext.Get()[i]=0;
		file.lpstrFilter=ext.Get();
	}
	if(type==XEF_OPEN)if(GetOpenFileName(&file))return(XE_STRING(path));
	if(type==XEF_SAVE)if(GetSaveFileName(&file))return(XE_STRING(path));
	return(XE_STRING());
}

void C_APPLICATION::ListFiles(FILELIST& list,char* path)
{
	WIN32_FIND_DATA find;
	HANDLE handle;
	list.Clear();
	handle=FindFirstFile(path,&find);
	if(handle!=INVALID_HANDLE_VALUE)
	{
		do
		{
			XE_ELEMENT_POINTER<XE_PAIR<XE_STRING,int> > elm=list.AddPointer(XE_PAIR<XE_STRING,int>());
			elm->Key=find.cFileName;
			if(find.dwFileAttributes&FILE_ATTRIBUTE_DIRECTORY)
				elm->Value=1;
			else
				elm->Value=0;
		}while(FindNextFile(handle,&find));
	}
	FindClose(handle);
}

XE_STRING C_APPLICATION::GetDirname(char* title,char* initdir,unsigned int flags,char* name,unsigned int id)
{
	HWND window;
	if(!name)
		window=NULL;
	else
	{
		C_ELM_WINDOW elm=Windows(XEF_FIND,name,id);
		if(!elm.IsEmpty())
			window=elm.iPointer()->GetWindow();
		else
			window=NULL;
	}
	XE_STRING result;
	BROWSEINFO bi={0};
	if(window)bi.hwndOwner=window;
	if(title)bi.lpszTitle=title;
	LPITEMIDLIST pidl=SHBrowseForFolder(&bi);
	if(pidl!=0)
	{
		char* path[MAX_PATH];
		if(SHGetPathFromIDList(pidl,(char*)path))result=(char*)path;
		IMalloc* imalloc=0;
		if(SUCCEEDED(SHGetMalloc(&imalloc)))
		{
			imalloc->Free(pidl);
			imalloc->Release();
		}
	}
	return(result);
}

void C_APPLICATION::MoveTo(XE_ACTOR* actor,XE_VECTOR point,double speed)
{
	if(!actor||speed<=0)return;
	double spd=min((point-actor->Position).Length(),speed);
	XE_VECTOR dir=(point-actor->Position).Normalize();
	actor->Position+=dir*XE_NUMBER(spd);
}

XE_VECTOR C_APPLICATION::MoveTo(XE_VECTOR pos,XE_VECTOR point,double speed)
{
	if(speed<=0)return(pos);
	double spd=min((point-pos).Length(),speed);
	XE_VECTOR dir=(point-pos).Normalize();
	return(pos+dir*XE_NUMBER(spd));
}

void C_APPLICATION::TorqueMoveTo(XE_ACTOR* actor,XE_VECTOR point,double speed,double distance)
{
	if(!actor||speed<=0)return;
	double len=(point-actor->Position).Length();
	if(len==0)return;
	double spd=min(len,speed);
	double f=min(len,distance)/distance;
	XE_VECTOR dir=(point-actor->Position).Normalize();
	actor->Position+=dir*XE_NUMBER(spd*f);
}

XE_VECTOR C_APPLICATION::TorqueMoveTo(XE_VECTOR pos,XE_VECTOR point,double speed,double distance)
{
	if(speed<=0)return(pos);
	double len=(point-pos).Length();
	if(len==0)return(pos);
	double spd=min(len,speed);
	double f=min(len,distance)/distance;
	XE_VECTOR dir=(point-pos).Normalize();
	return(pos+dir*XE_NUMBER(spd*f));
}

template<typename T>
T C_APPLICATION::Choose(int count,...)
{
	T result;
	int select=rand()%count;
	va_list vl;
	va_start(vl,count);
	for(int i=0;i<count;i++)
	{
		result=va_arg(vl,T);
		if(select==i)break;
	}
	va_end(vl);
	return(result);
}

bool C_APPLICATION::GoToScene(char* name,unsigned int id)
{
	if(!XE_EVENTS::Use().DoIt)return(false);
	C_ELM_SCENE elm=Scenes.Scenes(XEF_FIND,name,id);
	if(elm.IsEmpty())return(false);
	Scenes.Autonext=false;
	Scenes.Autoactivate=true;
	Scenes.Active=elm;
	XE_EVENTS::Use().DoIt=false;
	return(true);
}

bool C_APPLICATION::GoToScene(C_ELM_SCENE scene)
{
	if(!XE_EVENTS::Use().DoIt)return(false);
	if(scene.IsEmpty())return(false);
	Scenes.Autonext=false;
	Scenes.Autoactivate=true;
	Scenes.Active=scene;
	XE_EVENTS::Use().DoIt=false;
	return(true);
}

bool C_APPLICATION::NextScene()
{
	if(!XE_EVENTS::Use().DoIt)return(false);
	C_ELM_SCENE elm=Scenes.Active;
	elm.Next();
	if(elm.IsEmpty())return(false);
	Scenes.Autonext=false;
	Scenes.Autoactivate=true;
	Scenes.Active=elm;
	XE_EVENTS::Use().DoIt=false;
	return(true);
}

bool C_APPLICATION::PrevScene()
{
	if(!XE_EVENTS::Use().DoIt)return(false);
	C_ELM_SCENE elm=Scenes.Active;
	elm.Prev();
	if(elm.IsEmpty())return(false);
	Scenes.Autonext=false;
	Scenes.Autoactivate=true;
	Scenes.Active=elm;
	XE_EVENTS::Use().DoIt=false;
	return(true);
}

void C_APPLICATION::Exit()
{
	XE_EVENTS::Use().DoIt=false;
	Scenes.Active=C_ELM_SCENE();
}

#ifdef XE_COMPILE_PHOTON

void C_APPLICATION::SetColor(int color)
{
	XeSetState(XE_COLOR,GetFromRGBA(color,'R'),GetFromRGBA(color,'G'),GetFromRGBA(color,'B'),GetFromRGBA(color,'A'));
}

void C_APPLICATION::SetColor(XE_VECTOR color)
{
	XeSetState(XE_COLOR,color);
}

void C_APPLICATION::ClearSceneColor(int color)
{
	XeSetState(XE_SCENE_COLOR,GetFromRGBA(color,'R'),GetFromRGBA(color,'G'),GetFromRGBA(color,'B'),GetFromRGBA(color,'A'));
	Photon::XeRenderScene(XE_FLAG_COLORBUFF);
}

void C_APPLICATION::ClearSceneColor(XE_VECTOR color)
{
	XeSetState(XE_SCENE_COLOR,color);
	Photon::XeRenderScene(XE_FLAG_COLORBUFF);
}

void C_APPLICATION::Blend(XE_ESTATE src,XE_ESTATE dst,XE_ESTATE equation)
{
	XeSetState(XE_RENDER_BLEND_TYPE,src,dst);
	XeSetState(XE_RENDER_BLEND_EQUATION,equation);
}

void C_APPLICATION::Unblend()
{
	XeSetState(XE_RENDER_BLEND_TYPE,XE_RENDER_BLEND_SRC_ALPHA,XE_RENDER_BLEND_ONE_MINUS_SRC_ALPHA);
	XeSetState(XE_RENDER_BLEND_EQUATION,XE_RENDER_BLEND_ADD);
}

#endif /* XE_COMPILE_PHOTON */

XE_ACTOR2D* C_APPLICATION::Collision2D(XEF_ESTATE type,XE_ACTOR2D* act,char* desc,XE_VECTOR relative)
{
	if(!act)return(0);
	if(desc)
	{
		XE_FOREACH(XE_ACTOR*,Actors,elm)
		if(IsDescribedAs(elm.iClone(),desc))
		{
			if(type==XEF_BOX){if(act->CheckCollisionBox((XE_ACTOR2D*)elm.iClone(),relative))return((XE_ACTOR2D*)elm.iClone());}
			else
			if(type==XEF_CIRCLE){if(act->CheckCollisionCircle((XE_ACTOR2D*)elm.iClone()))return((XE_ACTOR2D*)elm.iClone());}
#ifdef XE_COMPILE_CHAOS
			else
			if(type==XEF_SHAPE){if(act->CheckCollisionShape((XE_ACTOR2D*)elm.iClone()))return((XE_ACTOR2D*)elm.iClone());}
#endif /* XE_COMPILE_CHAOS */
		}
		return(NULL);
	}
	else
	{
		XE_FOREACH(XE_ACTOR*,Actors,elm)
		{
			if(type==XEF_BOX){if(act->CheckCollisionBox((XE_ACTOR2D*)elm.iClone(),relative))return((XE_ACTOR2D*)elm.iClone());}
			else
			if(type==XEF_CIRCLE){if(act->CheckCollisionCircle((XE_ACTOR2D*)elm.iClone()))return((XE_ACTOR2D*)elm.iClone());}
#ifdef XE_COMPILE_CHAOS
			else
			if(type==XEF_SHAPE){if(act->CheckCollisionShape((XE_ACTOR2D*)elm.iClone()))return((XE_ACTOR2D*)elm.iClone());}
#endif /* XE_COMPILE_CHAOS */
		}
		return(NULL);
	}
}

void C_APPLICATION::Collision2Dlist(XE_ELEMENTS<XE_ACTOR2D*>& list,XEF_ESTATE type,XE_ACTOR2D* act,char* desc,XE_VECTOR relative)
{
	if(!act)return;
	list.Clear();
	if(desc)
	{
		XE_FOREACH(XE_ACTOR*,Actors,elm)
		if(IsDescribedAs(elm.iClone(),desc))
		{
			if(type==XEF_BOX){if(act->CheckCollisionBox((XE_ACTOR2D*)elm.iClone(),relative))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
			else
			if(type==XEF_CIRCLE){if(act->CheckCollisionCircle((XE_ACTOR2D*)elm.iClone()))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
#ifdef XE_COMPILE_CHAOS
			else
			if(type==XEF_SHAPE){if(act->CheckCollisionShape((XE_ACTOR2D*)elm.iClone()))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
#endif /* XE_COMPILE_CHAOS */
		}
	}
	else
	{
		XE_FOREACH(XE_ACTOR*,Actors,elm)
		{
			if(type==XEF_BOX){if(act->CheckCollisionBox((XE_ACTOR2D*)elm.iClone(),relative))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
			else
			if(type==XEF_CIRCLE){if(act->CheckCollisionCircle((XE_ACTOR2D*)elm.iClone()))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
#ifdef XE_COMPILE_CHAOS
			else
			if(type==XEF_SHAPE){if(act->CheckCollisionShape((XE_ACTOR2D*)elm.iClone()))list.AddPointer((XE_ACTOR2D*)elm.iClone());}
#endif /* XE_COMPILE_CHAOS */
		}
	}
}

unsigned int C_APPLICATION::MakeUnique()
{
	if(LastUnique==0xFFFFFFFF)LastUnique=0x00000000;
	return(LastUnique++);
}

void C_APPLICATION::SortActors()
{
	Actors.Sort(SortByOrder);
}

void C_APPLICATION::Register()
{
	ToRegister.Send(Actors);
}

void C_APPLICATION::Unregister()
{
	XE_ACTOR* temp;
	XE_FOREACH(XE_ACTOR*,ToUnregister,un)
	{
		temp=un.iClone();
		if(Actors.RemovePtr(un.iPointer()))
			delete temp;
	}
	ToUnregister.Clear();
}

XE_ACTOR* C_APPLICATION::RegisterActor(char* name,XE_ACTOR* newactor)
{
if(!newactor)return(0);
unsigned int id=Actors.MakeUnique();
ToRegister.Access(name,id)=newactor;
newactor->Desc.index=id;
return(newactor);
}

void C_APPLICATION::UnregisterActor(XE_ACTOR* actor)
{
if(!actor)return;
ToUnregister.AddPointer(actor);
}

void C_APPLICATION::Describe(XE_ACTOR* actor,char* text)
{
	strcpy_s(actor->Desc.string,sizeof(XE_S124I4)-sizeof(unsigned int)-1,text);
}

bool C_APPLICATION::IsDescribedAs(XE_ACTOR* actor,char* text)
{
	if(!strcmp(actor->Desc.string,text))return(true);
	return(false);
}

void C_APPLICATION::AlarmsProgress()
{
	XE_FOREACH(C_ALARM,Alarms,elm)
	{
		elm.iPointer()->Progress(XE_EVENTS::Use().DeltaTime);
	}
}

void C_APPLICATION::Update()
{
	if(GlobalUpdateProc.Proc)GlobalUpdateProc.Proc();
#ifdef XE_COMPILE_ETHER
	if(AutoSessionStage)XE_FOREACH(XE_SESSION,Sessions,elm)elm.iPointer()->Progress();
#endif /* XE_COMPILE_ETHER */
	if(AutoActorStage)XE_FOREACH(XE_ACTOR*,Actors,elm)elm.iClone()->Update();
#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH
	if(AutoParticleStage)XE_FOREACH(C_PARTICLES,Particles,elm)elm.iPointer()->Update(elm.iPointer()->Program);
#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */
}

#ifdef XE_COMPILE_PHOTON
#ifdef XE_COMPILE_CORE_MATH

void C_APPLICATION::Render()
{
	if(Views.Count()==0)
	{
		ViewActive.Unref();
		if(GlobalPreRenderProc.Proc)GlobalPreRenderProc.Proc();
		if(AutoParticleStage)
		XE_FOREACH(C_PARTICLES,Particles,elm)
		if(elm.iPointer()->Below&&!elm.iPointer()->Actor)
		elm.iPointer()->Draw();
		if(AutoActorStage)
		XE_FOREACH(XE_ACTOR*,Actors,elm)
		{
			if(AutoParticleStage)
			XE_FOREACH(C_PARTICLES,Particles,_elm)
			if(_elm.iPointer()->Below&&_elm.iPointer()->Actor==elm.iClone())
			_elm.iPointer()->Draw();
			elm.iClone()->Render();
			if(AutoParticleStage)
			XE_FOREACH(C_PARTICLES,Particles,_elm)
			if(!_elm.iPointer()->Below&&_elm.iPointer()->Actor==elm.iClone())
			_elm.iPointer()->Draw();
		}
#ifdef XE_COMPILE_CORE_DNA
#ifdef XE_COMPILE_CHAOS
		if(AutoDnaStage)XE_FOREACH(XE_DNA_MODEL,DNA,elm)elm.iPointer()->Draw();
#endif /* XE_COMPILE_CHAOS */
#endif /* XE_COMPILE_CORE_DNA */
		if(AutoParticleStage)
		XE_FOREACH(C_PARTICLES,Particles,elm)
		if(!elm.iPointer()->Below&&!elm.iPointer()->Actor)
		elm.iPointer()->Draw();
		if(GlobalPostRenderProc.Proc)GlobalPostRenderProc.Proc();
	}
	else
	XE_FOREACH(C_VIEW,Views,view)
	{
		if(view.iPointer()->Active)
		{
			ViewActive=view;
			view.iPointer()->Activate();
			if(GlobalPreRenderProc.Proc)GlobalPreRenderProc.Proc();
			if(AutoParticleStage)
			XE_FOREACH(C_PARTICLES,Particles,elm)
			if(elm.iPointer()->Below&&!elm.iPointer()->Actor)
			elm.iPointer()->Draw();
			if(AutoActorStage)
			XE_FOREACH(XE_ACTOR*,Actors,elm)
			{
				if(AutoParticleStage)
				XE_FOREACH(C_PARTICLES,Particles,_elm)
				if(_elm.iPointer()->Below&&_elm.iPointer()->Actor==elm.iClone())
				_elm.iPointer()->Draw();
				elm.iClone()->Render();
				if(AutoParticleStage)
				XE_FOREACH(C_PARTICLES,Particles,_elm)
				if(!_elm.iPointer()->Below&&_elm.iPointer()->Actor==elm.iClone())
				_elm.iPointer()->Draw();
			}
#ifdef XE_COMPILE_CORE_DNA
#ifdef XE_COMPILE_CHAOS
			if(AutoDnaStage)XE_FOREACH(XE_DNA_MODEL,DNA,elm)elm.iPointer()->Draw();
#endif /* XE_COMPILE_CHAOS */
#endif /* XE_COMPILE_CORE_DNA */
			if(AutoParticleStage)
			XE_FOREACH(C_PARTICLES,Particles,elm)
			if(!elm.iPointer()->Below&&!elm.iPointer()->Actor)
			elm.iPointer()->Draw();
			if(GlobalPostRenderProc.Proc)GlobalPostRenderProc.Proc();
			C_VIEW::Unactivate();
		}
	}
	ViewActive.Unref();
}

#endif /* XE_COMPILE_CORE_MATH */
#endif /* XE_COMPILE_PHOTON */

void C_APPLICATION::Progress()
{
	if(XE_EVENTS::CanUpdate())Update();
#ifdef XE_COMPILE_PHOTON
	if(AutoGuiStage)
	{
		if(XE_EVENTS::CanUpdate())if(GUI.GetWindow())GUI.Update();
		GUI.Progress();
		GUI.Unregister();
		GUI.Synchronize();
	}
#ifdef XE_COMPILE_CORE_MATH
	if(XE_EVENTS::CanRender())Render();
#endif /* XE_COMPILE_CORE_MATH */
	if(AutoGuiStage)if(XE_EVENTS::CanRender())if(GUI.GetWindow())GUI.Draw(GUI.GetWindow()->Layered);
#endif /* XE_COMPILE_PHOTON */
	if(AutoActorStage)
	{
		Register();
		Unregister();
	}
	if(AutoAlarmStage)AlarmsProgress();
}

#endif /* XE_CANIMP */

} /* namespace: Com */
} /* namespace: XeCore */

#endif /* XE_COMPONENT_APP_H */
